/*
 * Copyright 2020 by Beat Hörmann
 * Source code licensed under "Lida License"
 */
package acdp.design;

import javax.crypto.Cipher;

/**
 * Declares methods used to encrypt and decrypt data in a database.
 * <p>
 * The creation and initialization of {@link Cipher} instances for encryption
 * and decryption is under control of the client.
 * The client can choose whatever type of cryptographic cipher he or she
 * prefers&mdash;with a single restriction:
 * For a WR database the cipher must encrypt data in the unit of a single byte.
 * (Such a cipher is sometimes called a <em>byte-oriented stream cipher</em> or
 * a block cipher that <em>is turned</em> into a byte-oriented stream cipher.
 * See the description of the {@code Cipher} class for details.)
 * No such restriction exists for an RO database.
 * <p>
 * Of course, this interface must only be implemented if cryptographic
 * functionality is required at all.
 *
 * @author Beat Hörmann
 */
public interface ICipherFactory {
	/**
	 * Returns a new cipher instance to be used by ACDP for encrypting/decrypting
	 * data in a database.
	 * <p>
	 * If the database is a WR database then the returned cipher must be a
	 * byte-oriented stream cipher or must behave that way.
	 * Otherwise the database is an RO database and the returned cipher may be a
	 * block cipher encrypting data in units larger than a single byte.
	 * <p>
	 * This method may throw various kinds of exceptions depending on the
	 * implementation.
	 * See the description of the {@code getInstance} methods of the {@link
	 * Cipher} class for any details.
	 * 
	 * @return The created uninitialized cipher instance, never {@code null}.
	 *         
	 * @throws Exception If creating the cipher fails.
	 */
	Cipher createCipher() throws Exception;
	
	/**
	 * Initializes the specified cipher, earlier created by a call to the {@link
	 * #createCipher} method.
	 * <p>
	 * This method may throw various kinds of exceptions depending on the
	 * implementation.
	 * See the description of the {@code init} methods of the {@link Cipher}
	 * class for any details.
	 * 
	 * @param  cipher The cipher to initialize, not allowed to be {@code null}.
	 * @param  encrypt Initialize the cipher for encryption ({@code true}) or
	 *         decryption ({@code false}).
	 *        
	 * @throws Exception If initializing the cipher fails.
	 */
	void initCipher(Cipher cipher, boolean encrypt) throws Exception;
}
