/*
 * Copyright 2020 by Beat Hörmann
 * Source code licensed under "Lida License"
 */
package acdp.internal.types;

import java.util.Objects;

import acdp.types.Type;

/**
 * The super class of all column types.
 *
 * @author Beat Hörmann
 */
public abstract class Type_ implements Type {
	/**
	 * Determines if the specified type descriptor describes a built-in column
	 * type.
	 * The type descriptor describes a built-in column type if and only if the
	 * type descriptor does not describe an array type and starts with a lower
	 * case character <em>or</em> the part of the type descriptor that
	 * describes the type of the elements of the array starts with a lower case
	 * character.
	 * 
	 * @param  typeDesc The type descriptor, not allowed to be {@code null} and
	 *         not allowed to be an empty string.
	 * @return The boolean value {@code true} if the type descriptor describes
	 *         a built-in type, {@code false} otherwise.
	 */
	public static final boolean isBuiltInType(String typeDesc) {
		final char firstChar = typeDesc.charAt(0);
		if (firstChar == ArrayType_.KIND)
			return Character.isLowerCase(ArrayType_.getTypeDescET(typeDesc).
																							charAt(0));
		else {
			return Character.isLowerCase(firstChar);
		}
	}
	
	/**
	 * The storage scheme, never {@code null}.
	 */
	protected final Scheme scheme;
	
	/**
	 * The constructor.
	 * 
	 * @param  scheme The storage scheme of the type, not allowed to be
	 *         {@code null}.
	 * @throws NullPointerException If {@code scheme} is {@code null}.
	 */
	protected Type_(Scheme scheme) throws NullPointerException {
		this.scheme = Objects.requireNonNull(scheme, "The value for the "+
														"storage is not allowed to be null.");
	}
	
	@Override
	public final Scheme scheme() {
		return scheme;
	}
}
