/*
 * Created on 30.06.2025.
 *
 * Copyright 2025 TRIANGULUM AG, Adlerfeldstrasse 49, 4402 Frenkendorf,
 * Switzerland. All rights reserved.
 */
package acdp.tools;

import static java.nio.file.StandardOpenOption.CREATE_NEW;
import static java.nio.file.StandardOpenOption.WRITE;

import java.io.IOException;
import java.nio.file.Files;
import java.nio.file.Path;
import java.util.ArrayList;
import java.util.Arrays;
import java.util.HashMap;
import java.util.HashSet;
import java.util.Iterator;
import java.util.List;
import java.util.Map;
import java.util.Map.Entry;
import java.util.Set;

import acdp.ColVal;
import acdp.Column;
import acdp.Database;
import acdp.Information.ColumnInfo;
import acdp.Row;
import acdp.Table;
import acdp.design.ICipherFactory;
import acdp.exceptions.CreationException;
import acdp.exceptions.IOFailureException;
import acdp.internal.Database_;
import acdp.internal.Table_;
import acdp.internal.store.wr.WRStore;
import acdp.misc.ACDP;
import acdp.misc.Layout;

/**
 * Given an RO database and an optional layout file that is "compatible" with
 * the RO database, this class creates a WR database based on the layout file
 * and copies the content of the RO database to the newly created WR database.
 * If no layout file is provided then the layout of the WR database will be
 * derived from the layout of the RO database, with some fields set to estimated
 * values.
 *
 * @author Beat Hörmann
 */
public final class ROToWR {
	/**
	 * Computes the set of names of those tables that are referenced by at least
	 * one column of any table in the database.
	 * 
	 * @param  tables The tables of the database.
	 * 
	 * @return The names of the referenced tables.
	 */
	private final Set<String> refdTables(Table[] tables) {
		final Set<String> refdTables = new HashSet<>();
		for (Table table : tables) {
			for (ColumnInfo ci : table.info().columnInfos()) {
				final String refdTable = ci.refdTable();
				if (refdTable != null) {
					refdTables.add(refdTable);
				}
			}
		}
		return refdTables;
	}
	
	/**
	 * Derives a WR layout from the specified RO database.
	 * 
	 * @param  roDB The RO database.
	 * 
	 * @return The derived WR layout.
	 */
	private final Layout deriveLayout(Database roDB) {
		// Deeply copy layout of the RO database.
		final Layout layout = new Layout(((Database_) roDB).layout());
		
		// Add "forceWrite" and "recFile" entries.
		layout.add(Database_.lt_forceWriteCommit, Database_.off);
		layout.add(Database_.lt_recFile, "rec");
		
		final Table[] tables = roDB.getTables();
		final Set<String> refdTables = refdTables(tables);
		final Layout tablesLayout = layout.getLayout(Database_.lt_tables);
		
		// With the exception of the store layout, the table layouts are the
		// same for RO and WR databases. Create WR store layout and replace
		// existing RO store layout with newly created WR store layout.
		for (Table table : tables) {
			final String name = table.name();
			final Layout storeLayout = WRStore.createLayout(
															refdTables.contains(name),
															((Table_) table).tableDef(), name);
			// Replace store layout.
			tablesLayout.getLayout(name).replace(Table_.lt_store, storeLayout);
		}
		
		return layout;
	}
	
	/**
	 * As a result of this method, the specified directory contains the file
	 * named "layout" which contains the layout of the WR database to be created.
	 * 
	 * @param  roDB The RO database.
	 * @param  wrDbDirPath The directory where to store the WR database; not
	 *         allowed to be {@code null}.
	 *         The directory will be created if it does not exist.
	 * @param  layoutFile The layout file or {@code null}.
	 * 
	 * @return The layout file of the WR database to be created.
	 * 
	 * @throws NullPointerException If {@code wrDbDirPath} is {@code null}.
	 * @throws IOFailureException If the directory pointed to by the {@code
	 *         wrDbDirPath} argument already contains a file with the name
	 *         "layout" which is not equal to the specified layout file or if
	 *         another I/O error occurs.
	 */
	private final Path establishLayoutFile(Database roDB, Path wrDbDirPath,
																			Path layoutFile) throws
												NullPointerException, IOFailureException {
		final Path ltFile = wrDbDirPath.resolve("layout");
		try {
			Files.createDirectories(wrDbDirPath);
			if (layoutFile == null)
				deriveLayout(roDB).toFile(ltFile, null, CREATE_NEW, WRITE);
			else {
				if (!layoutFile.equals(ltFile)) {
					Files.copy(layoutFile, ltFile);
				}
				// Ensure that the DB name registered in the layout file is equal to
				// the name of the RO database.
				final Layout layout = Layout.fromFile(ltFile);
				final String name = roDB.name();
				if (!layout.getString(Database_.lt_name).equals(name)) {
					layout.replace(Database_.lt_name, name).save();
				}
			}
		} catch (IOException e) {
			throw new IOFailureException(e);
		}
		return ltFile;
	}
	
	/**
	 * Creates an empty WR database based either on the layout stored in the
	 * specified layout file or on a layout derived from the layout of the
	 * specified RO database, depending on whether the {@code layoutFile}
	 * argument is equal to {@code null}.
	 * <p>
	 * The backing files of the newly created WR database as well as the layout
	 * file with the name "layout" are stored in the specified directory.
	 * 
	 * @param  roDB The RO database.
	 * @param  wrDbDirPath The directory where to store the WR database; not
	 *         allowed to be {@code null} and must exist.
	 *         The directory will be created if it does not exist.
	 * @param  layoutFile The layout file or {@code null}.
	 *         
	 * @return The layout file of the created WR database.
	 * 
	 * @throws NullPointerException If {@code wrDbDirPath} is {@code null}.
	 * @throws IOFailureException If the directory pointed to by the {@code
	 *         wrDbDirPath} argument already contains a file with the name
	 *         "layout" which is not equal to the specified layout file or if at
	 *         least one of the backing files of the WR database to be created
	 *         already exists or if another I/O error occurs.
	 */
	private final Path createDB(Database roDB, Path wrDbDirPath,
																			Path layoutFile) throws
												NullPointerException, IOFailureException {
		// Establish layout file of the WR database to be created.
		final Path ltFile = establishLayoutFile(roDB, wrDbDirPath, layoutFile);
		// Create the backing DB files according to the layout.
		ACDP.createDBFiles(ltFile);
		
		return ltFile;
	}
	
	/**
	 * Whether the specified table can be copied.
	 * 
	 * A table can be copied if none of its columns references a table that
	 * has not yet been copied.
	 * 
	 * @param  table The table to check.
	 * @param  copied The tables that have already been copied.
	 * 
	 * @return The boolean value {@code true} if and only if this table can be
	 *         copied.
	 */
	private final boolean canBeCopied(Table table, Set<String> copied) {
		return Arrays.stream(table.info().columnInfos()).allMatch(ci ->
							ci.refdTable() == null || copied.contains(ci.refdTable()));
	}
	
	/**
	 * Copies the rows of the specified RO table to the specified WR table.
	 * Of course, both tables must have the same table definition.
	 * 
	 * @param roTable The RO table.
	 * @param wrTable The WR table.
	 */
	private final void copyRows(Table roTable, Table wrTable) {
		final int nofCols = roTable.getColumns().length;
		final Object[] values = new Object[nofCols];
		for (Row roRow : roTable) {
			for (int i = 0; i < nofCols; i++) {
				values[i] = roRow.get(i);
			}
			wrTable.insert(values);
		}
	}
	
	/**
	 * Copies the rows of those tables contained in the specified array of tables
	 * that have not yet been copied and that {@linkplain #canBeCopied can be
	 * copied}.
	 * 
	 * @param  tables The tables to copy.
	 * @param  wrDB The WR database.
	 * @param  copied The tables that have already been copied.
	 * 
	 * @return The number of copied tables.
	 */
	private final int copyTables(Table[] tables, Database wrDB,
																			Set<String> copied) {
		int nofCopied = 0;
		for (Table table : tables) {
			final String name = table.name();
			if (!copied.contains(name)) {
				if (canBeCopied(table, copied)) {
					copyRows(table, wrDB.getTable(name));
					copied.add(name);
					nofCopied++;
				}
			}
		}
		return nofCopied;
	}
	
	
	/**
	 * Partially copies the rows of the specified RO table to the specified WR
	 * table.
	 * Of course, both tables must have the same table definition.
	 * As a side effect, this methods puts the RO table into the {@code
	 * unfinished} map along with its unfinished WR columns.
	 * 
	 * @param roTable The RO table.
	 *        The table has at least one column that cannot be copied.
	 * @param wrTable The WR table.
	 * @param copied The tables that have already been copied.
	 * @param unfinished Maps a table to its unfinished WR columns.
	 */
	private final void partiallyCopyRows(Table roTable, Table wrTable,
													Set<String> copied,
													Map<Table, List<Column<?>>> unfinished) {
		final Column<?>[] columns = roTable.getColumns();
		final int nofCols = columns.length;
		final ColumnInfo[] colInfos = roTable.info().columnInfos();
		
		final List<Column<?>> unfinishedCols = new ArrayList<>();
		
		// Create a boolean vector that indicates whether the column with index i
		// can be copied. At the same time register the unfinished columns.
		final boolean[] canBeCopied = new boolean[nofCols];
		for (int i = 0; i < nofCols; i++) {
			final String refdTable = colInfos[i].refdTable();
			final boolean colCanBeCopied = refdTable == null || copied.contains(
																							refdTable);
			if (!colCanBeCopied) {
				unfinishedCols.add(wrTable.getColumn(columns[i].name()));
			}
			canBeCopied[i] = colCanBeCopied;
		}
		
		unfinished.put(roTable, unfinishedCols);
		
		// Copy all rows. Insert null for columns that cannot be copied.
		final Object[] values = new Object[nofCols];
		Arrays.fill(values, null);
		for (Row roRow : roTable) {
			for (int i = 0; i < nofCols; i++) {
				if (canBeCopied[i]) {
					values[i] = roRow.get(i);
				}
			}
			wrTable.insert(values);
		}
	}
	
	/**
	 * Partially copies the rows of the specified table and register it along
	 * with its unfinished columns into the specified map.
	 * 
	 * @param table The table.
	 * @param wrDB The WR database.
	 * @param copied The tables that have already been copied.
	 * @param unfinished Maps a table to its unfinished WR columns.
	 */
	private final void partiallyCopyTable(Table table, Database wrDB,
													Set<String> copied,
													Map<Table, List<Column<?>>> unfinished) {
		final String name = table.name();
		//  Partially copy the rows of the table.
		partiallyCopyRows(table, wrDB.getTable(name), copied, unfinished);
		// Register the table as being copied.
		copied.add(name);
	}
	
	
	/**
	 * Exhaustively copies the tables that have not yet been copied.
	 * 
	 * @param  tables All the tables of the RO database.
	 * @param  wrDB The WR database.
	 * @param  copied The tables that have already been copied.
	 */
	private final void phase2(Table[] tables, Database wrDB,
																			Set<String> copied) {
		final Map<Table, List<Column<?>>> unfinished = new HashMap<>();
		
		final int nofTables = tables.length;
		do {
			// Reduce candidate tables to tables that have not yet been copied.
			// Note that each of these tables contains at least one column that
			// cannot be copied. These tables can therefore only be "partially"
			// copied.
			tables = Arrays.stream(tables).filter(t -> !copied.contains(
															t.name())).toArray(Table[]::new);
			// Copy the rows of the first table setting null for columns that
			// cannot be copied. I couldn't find a significant better heuristic
			// than just choosing the first one.
			partiallyCopyTable(tables[0], wrDB, copied, unfinished);
			// After one of these tables has partially been copied, there may
			// exist one or more tables that can be "fully" copied. Fully copy
			// those tables until no more tables can be fully copied.
			while (copyTables(tables, wrDB, copied) > 0) {}
		} while (copied.size() < nofTables);
		// All tables have been copied, at least one of them only "partially".
		
		// Finish: Update rows on unfinished columns.
		for (Entry<Table, List<Column<?>>> entry : unfinished.entrySet()) {
			final Table roTable = entry.getKey();
			final Table wrTable = wrDB.getTable(roTable.name());
			final List<Column<?>> wrCols = entry.getValue();
			final int nofCols = wrCols.size();
			final Column<?>[] roCols = wrCols.stream().map(c -> roTable.getColumn(
															c.name())).toArray(Column[]::new);
			
			final ColVal<?>[] colVals = new ColVal[nofCols];
			final Iterator<Row> roTableIt = roTable.iterator(roCols);
			while (roTableIt.hasNext()) {
				final Row row = roTableIt.next();
				for (int i = 0; i < nofCols; i++) {
					@SuppressWarnings("unchecked")
					final Column<Object> column = (Column<Object>) wrCols.get(i);
					colVals[i] = column.value(row.get(i));
				}
				wrTable.update(row.getRef(), colVals);
			}
		}
	}
	/**
	 * See {@linkplain #run}
	 * 
	 * @param  roDB The RO database.
	 * @param  wrDbDirPath The directory where to store the resulting WR
	 *         database; not allowed to be {@code null} and must exist.
	 *         The directory is not allowed to contain a file with a name equal
	 *         to "layout" nor any files with names equal to the names of any
	 *         backing files of the WR database to be created.
	 *         The directory will be created if it does not exist.
	 * @param  layoutFile The layout file of the WR database to be created or
	 *         {@code null}.
	 *         This is typically the layout file of the WR database used to
	 *         create the RO database.
	 *         If the layout file is contained in the directory pointed to by the
	 *         {@code wrDbDirPath} argument and its name is equal to "layout"
	 *         then this layout file will be the layout file of the WR database
	 *         to be created.
	 *         If the value is {@code null} then the layout will be derived from
	 *         the layout of the RO database.
	 *         See the method description for more information.
	 * @param  cf The cipher factory or {@code null}.
	 *         The cipher factory is used for reading the RO database as well as
	 *         for writing the WR database.
	 *        
	 * @throws NullPointerException If {@code wrDbDirPath} is {@code null}.
	 * @throws IOFailureException If the directory pointed to by the {@code
	 *         wrDbDirPath} argument already contains a file with the name
	 *         "layout" which is not equal to the specified layout file or if at
	 *         least one of the backing files of the WR database to be created
	 *         already exists or if another I/O error occurs.
	 */
	private final void convert(Database roDB, Path wrDbDirPath, Path layoutFile,
																		ICipherFactory cf) throws
												NullPointerException, IOFailureException {
		// Create the WR database.
		layoutFile = createDB(roDB, wrDbDirPath, layoutFile);
		try (Database wrDB = Database.open(layoutFile, -1, false, cf)) {
			Table[] tables = roDB.getTables();
			final Set<String> copied = new HashSet<>(tables.length * 4 / 3 + 1);
			
			// Phase 1: Copy the tables until no more tables can be copied.
			while (copyTables(tables, wrDB, copied) > 0) {}
			
			// Either all tables are copied or there are tables that cannot be
			// copied due to one or more tables that form at least one directed
			// cyclic reference graph.
			if (copied.size() < tables.length) {
				// Phase2: There are cycles!
				phase2(tables, wrDB, copied);
			}
		}
	}
	
	/**
	 * Creates a WR database and copies all rows from all tables of the specified
	 * RO database to the newly created WR database.
	 * The newly created WR datastore is stored in the directory with a path
	 * equal to the {@code wrDbDirPath} argument and has a layout file with a
	 * name equal to "layout".
	 * If the {@code layoutFile} argument is not {@code null} then the WR
	 * database is created according to the layout saved in the layout file.
	 * Otherwise, the WR database layout will be derived from the RO database
	 * layout, with some fields set to estimated values ​​that you may want to
	 * adjust later.
	 * (See the corresponding subsection in the description of the {@linkplain
	 * Setup Setup Tool}.)
	 * <p>
	 * Note that existing references to rows of a particular table of the RO
	 * database are valid in the created WR database.
	 * <p>
	 * If the specified layout file does not contain a valid layout for a WR
	 * database or does not reflect the structure of the RO database then this
	 * method may throw an exception that is different from the listed
	 * exceptions.
	 * 
	 * @param  roDbFilePath The path of the RO database file; not allowed to be
	 *         {@code null}.
	 * @param  wrDbDirPath The directory where to store the resulting WR
	 *         database; not allowed to be {@code null}.
	 *         The directory is not allowed to contain a file with a name equal
	 *         to "layout" nor any files with names equal to the names of any
	 *         backing files of the WR database to be created.
	 *         The directory will be created if it does not exist.
	 * @param  layoutFile The layout file of the WR database to be created or
	 *         {@code null}.
	 *         This is typically the layout file of the WR database used to
	 *         create the RO database.
	 *         If the layout file is contained in the directory pointed to by the
	 *         {@code wrDbDirPath} argument and its name is equal to "layout"
	 *         then this layout file will be the layout file of the WR database
	 *         to be created.
	 *         If the value is {@code null} then the layout will be derived from
	 *         the layout of the RO database.
	 *         See the method description for more information.
	 * @param  cf The cipher factory or {@code null}.
	 *         The cipher factory is used for reading the RO database as well as
	 *         for writing the WR database.
	 *        
	 * @throws NullPointerException If {@code roDbFilePath} or {@code
	 *         wrDbDirPath} are {@code null}.
	 * @throws IOFailureException If the directory pointed to by the {@code
	 *         wrDbDirPath} argument already contains a file with the name
	 *         "layout" which is not equal to the specified layout file or if at
	 *         least one of the backing files of the WR database to be created
	 *         already exists or if another I/O error occurs.
	 * @throws CreationException If the RO database can't be opened due to any
	 *         reason including the content of the {@code roDbFilePath} file not
	 *         being an RO database or problems regarding encryption.
	 */
	public static final void run(Path roDbFilePath, Path wrDbDirPath,
												Path layoutFile, ICipherFactory cf) throws
																				NullPointerException,
																				IOFailureException,
																				CreationException {
		try (Database roDB = Database.open(roDbFilePath, -3, true, cf)) {
			new ROToWR().convert(roDB, wrDbDirPath, layoutFile, cf);
		}
	}
	
	/**
	 * Prevent object construction.
	 */
	private ROToWR() {
	}
}
