package example.pl;

import static example.db.ContactTable.CITY;
import static example.db.ContactTable.COUNTRY;
import static example.db.ContactTable.EMAIL;
import static example.db.ContactTable.PHONE_NUMBER;
import static example.db.ContactTable.POSTAL_CODE;
import static example.db.ContactTable.STATE;
import static example.db.ContactTable.STREET;
import static example.db.EduDB.CONTACT_TABLE;
import static example.db.EduDB.COURSE_TABLE;
import static example.db.EduDB.STUDENT_TABLE;
import static example.db.EduDB.TEACHER_TABLE;

import java.math.BigInteger;
import java.nio.file.Path;
import java.util.ArrayList;
import java.util.List;
import java.util.Objects;

import acdp.ColVal;
import acdp.ReadZone;
import acdp.Ref;
import acdp.Row;
import acdp.Unit;
import example.db.CourseTable;
import example.db.EduDB;
import example.db.StudentTable;
import example.db.TeacherTable;

/**
 *
 *
 * @author Beat Hörmann
 */
public final class EduPL implements AutoCloseable {
	private final EduDB db;
	
	public EduPL(Path mainFile, int opMode, boolean writeProtect,
																		int consistencyNumber) {
		db = new EduDB(mainFile, opMode, writeProtect, consistencyNumber);
	}
	
	public final void persistCourse(String id, String name) {
		final Ref ref = COURSE_TABLE.indexGet(id);
		if (ref == null)
			COURSE_TABLE.insert(id, name, null);
		else {
			COURSE_TABLE.update(ref, CourseTable.NAME.value(name));
		}
	}
	
	private final Ref insertContact(Contact contact) {
		return CONTACT_TABLE.insert(contact.getPhoneNumber(), contact.getEmail(),
							contact.getStreet(), contact.getCity(), contact.getState(),
							contact.getPostalCode(), contact.getCountry());
	}
	
	private final void updateContact(Ref ref, Contact contact) {
		final Row stored = CONTACT_TABLE.get(ref);
		
		final List<ColVal<?>> colVals = new ArrayList<>();
		if (!Objects.equals(contact.getPhoneNumber(), stored.get(PHONE_NUMBER))) {
			colVals.add(PHONE_NUMBER.value(contact.getPhoneNumber()));
		}
		if (!contact.getEmail().equals(stored.get(EMAIL))) {
			colVals.add(EMAIL.value(contact.getEmail()));
		}
		if (!contact.getStreet().equals(stored.get(STREET))) {
				colVals.add(STREET.value(contact.getStreet()));
		}
		if (!contact.getCity().equals(stored.get(CITY))) {
			colVals.add(CITY.value(contact.getCity()));
		}
		if (!contact.getState().equals(stored.get(STATE))) {
			colVals.add(STATE.value(contact.getState()));
		}
		if (Integer.compare(contact.getPostalCode(),
															stored.get(POSTAL_CODE)) != 0) {
			colVals.add(POSTAL_CODE.value(contact.getPostalCode()));
		}
		if (!Objects.equals(contact.getCountry(), stored.get(COUNTRY))) {
			colVals.add(COUNTRY.value(contact.getCountry()));
		}
		
		CONTACT_TABLE.update(ref, colVals.toArray(new ColVal<?>[colVals.size()]));
	}
	
	public final void persistStudent(int studentNumber, String name,
													Contact contact, List<String> courses) {
		// Get row references of courses.
		final Ref[] courseRefs = new Ref[courses.size()];
		for (int i = 0; i < courseRefs.length; i++) {
			courseRefs[i] = COURSE_TABLE.get(COURSE_TABLE.indexGet(
																		courses.get(i))).getRef();
		}
		
		final Ref ref = STUDENT_TABLE.indexGet(studentNumber);
		if (ref == null)
			STUDENT_TABLE.insert(studentNumber, name, insertContact(contact),
																						courseRefs);
		else {
			updateContact(STUDENT_TABLE.getValue(ref, StudentTable.CONTACT),
																						contact);
			STUDENT_TABLE.update(ref, StudentTable.NAME.value(name),
													StudentTable.COURSES.value(courseRefs));
		}
	}
	
	public final void persistTeacher(String name, BigInteger salary,
													Contact contact, List<String> courses) {
		// Get row references of courses.
		final Ref[] courseRefs = new Ref[courses.size()];
		for (int i = 0; i < courseRefs.length; i++) {
			courseRefs[i] = COURSE_TABLE.get(COURSE_TABLE.indexGet(
																		courses.get(i))).getRef();
		}
		
		Ref ref = TEACHER_TABLE.indexGet(name);
		if (ref == null)
			ref = TEACHER_TABLE.insert(name, salary, insertContact(contact),
																						courseRefs);
		else {
			updateContact(TEACHER_TABLE.getValue(ref, TeacherTable.CONTACT),
																							contact);
			TEACHER_TABLE.update(ref, TeacherTable.SALARY.value(salary),
											TeacherTable.COURSES.value(courseRefs));
		}
		// ref != null
		
		// Update the link from course to teacher.
		for (Ref courseRef : courseRefs) {
			COURSE_TABLE.update(courseRef, CourseTable.TEACHER.value(ref));
		}
	}
	
	public final void deleteCourse(String id) {
		COURSE_TABLE.delete(id);
	}
	
	public final void deleteStudent(int studentNumber) {
		final Ref contactRef = STUDENT_TABLE.getValue(
												STUDENT_TABLE.indexGet(studentNumber),
												StudentTable.CONTACT);
		STUDENT_TABLE.delete(studentNumber);
		CONTACT_TABLE.delete(contactRef);
	}
	
	public final void deleteTeacher(String name) {
		final Row row = TEACHER_TABLE.get(TEACHER_TABLE.indexGet(name),
										TeacherTable.CONTACT, TeacherTable.COURSES);
		// Release link from course to teacher.
		for (Ref courseRef : row.get(TeacherTable.COURSES)) {
			COURSE_TABLE.update(courseRef, CourseTable.TEACHER.value(null));
		}
		
		TEACHER_TABLE.delete(name);
		CONTACT_TABLE.delete(row.get(TeacherTable.CONTACT));
	}
	
	public final Course getCourse(String id) {
		final Row row = COURSE_TABLE.get(COURSE_TABLE.indexGet(id));
		return new Course(row.get(CourseTable.ID),
								row.get(CourseTable.NAME),
								row.get(CourseTable.TEACHER));
	}
	
	public final Student getStudent(int studentNumber) {
		final Row row = STUDENT_TABLE.get(STUDENT_TABLE.indexGet(studentNumber));
		return new Student(row.get(StudentTable.STUDENT_NUMBER),
								 row.get(StudentTable.NAME),
								 row.get(StudentTable.CONTACT),
								 row.get(StudentTable.COURSES));
	}
	
	public final Teacher getTeacher(String name) {
		final Row row = TEACHER_TABLE.get(TEACHER_TABLE.indexGet(name));
		return new Teacher(row.get(TeacherTable.NAME),
								 row.get(TeacherTable.SALARY),
								 row.get(TeacherTable.CONTACT),
								 row.get(TeacherTable.COURSES));
	}
	
	public final Unit openUnit() {
		return db.openUnit();
	}
	
	public final ReadZone openReadZone() {
		return db.openReadZone();
	}

	@Override
	public void close() {
		db.close();
	}
}
