/*
 * Copyright 2020 by Beat Hörmann
 * Source code licensed under "Lida License"
 */
package acdp.exceptions;

import java.nio.file.Path;

import acdp.design.CustomTable;
import acdp.internal.Database_;

/**
 * Thrown if the value of the consistency number given at the time a strongly
 * typed database is {@linkplain acdp.design.CustomDatabase#open(Path, int,
 * boolean, ICipherFactory, int, CustomTable...) opened} differs from the value
 * of the consistency number stored in the database layout.
 * Signals that the logic of a strongly typed database is inconsistent with the
 * data stored in that database.
 * <p>
 * The different consistency numbers can be retrieved invoking the {@link
 * #logicCN()} and {@link #dataCN()} methods.
 *
 * @author Beat Hörmann
 */
public final class ConsistencyException extends ACDPException {
	private static final long serialVersionUID = 2563395674421102623L;
	
	/**
	 * The consistency number of the custom database logic.
	 */
	private final int logicCN;
	/**
	 * The consistency number as stored in the database.
	 */
	private final int dataCN;

	/**
    * The constructor.
    * 
    * @param db The database.
    * @param logicCN The consistency number of the logic.
    * @param dataCN The consistency number of the data.
    */
	public ConsistencyException(Database_ db, int logicCN, int dataCN) {
		super(db, "The logic of the database is inconsistent with its data. " +
										"Consistency number of logic: " + logicCN +
										", consistency number of data: " + dataCN + ".");

		this.logicCN = logicCN;
		this.dataCN = dataCN;
	}
	
	/**
	 * Returns the consistency number of the custom database logic.
	 * 
	 * @return The consistency number of the custom database logic.
	 */
	public final int logicCN() {
		return logicCN;
	}
	
	/**
	 * Returns the consistency number as stored in the database.
	 * 
	 * @return The consistency number as stored in the database.
	 */
	public final int dataCN() {
		return dataCN;
	}
}
