/*
 * Copyright 2020 by Beat Hörmann
 * Source code licensed under "Lida License"
 */
package acdp.types;

import java.nio.charset.Charset;

/**
 * The column type analogon of a Java {@code String} type.
 * In contrast to the Java {@code String} type this type may be specified such
 * that the {@code null} value is forbidden.
 * <p>
 * Besides the property that decides if the {@code null} value is allowed or
 * not, the string type has three additional properties: The {@linkplain
 * Type.Scheme storage scheme}, the <em>limit</em> and the <em>charset</em>.
 * All have an influence on the maximum number of characters in a string value
 * as follows:
 * 
 * <h2>Inrow Storage Scheme</h2>
 * If the chosen charset is a single byte character set then the maximum number
 * of characters in a string is equal to the value of the limit.
 * If the chosen charset is not a single byte character set then the maximum
 * number of characters may be less than the value of the limit.
 * 
 * <h2>Outrow Storage Scheme</h2>
 * If the chosen charset is a single byte character set then the maximum number
 * of characters in a string is equal to {@code n} = {@code min}(256<sup>{@code
 * limit}</sup> - 1, {@code Integer.MAX_VALUE}), e.g., 255 for a {@code limit}
 * equal to 1.
 * If the chosen charset is not a single byte character set then the maximum
 * number of characters may be less than {@code n} characters.
 * <p>
 * There should be no need for clients to implement this interface.
 *
 * @author Beat Hörmann
 */
public interface StringType extends Type {
	/**
	 * Returns the character set of this string type.
	 * 
	 * @return The character set, never {@code null}.
	 */
	Charset charset();
	
	/**
	 * Returns the limit of this string type.
	 * 
	 * @return The limit, greater than or equal to 1.
	 */
	int limit();
}
